#f!/usr/bin/perl
# Copyright (C) 2015 Apple Bottom

# Parses and aggregates logfiles generated by Calcyman's agpnano soup
# searcher for Conway's Game of Life, and generates statistics.

#-------------------------------------------------------------------------------
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see [http://www.gnu.org/licenses/].
#-------------------------------------------------------------------------------

# the usual preliminaries
use v5.14;
use strict;
use warnings;
use English;

# core modules
use File::Basename;
use File::Path                qw/make_path/;
use Getopt::Long;
use List::Util                qw/max min sum0/;
use Term::ANSIColor;
use Time::HiRes               qw/time/;

# non-core modules
use DateTime;						# Ubuntu: libdatetime-perl
use Date::Range;
use Date::Simple;
use GD;							# Ubuntu: libgd-perl
use GD::Graph::mixed;					# Ubuntu: libgd-graph-perl
use List::Compare::Functional qw/get_unique get_union/;	# Ubuntu: liblist-compare-perl
use List::MoreUtils           qw/uniq/;                 # Ubuntu: liblist-moreutils-perl
use Math::Prime::Util v0.46   qw/hammingweight/;	# Ubuntu: libmath-prime-util-perl
use Term::ReadKey;					# Ubuntu: libterm-readkey-perl

# install fewer from CPAN, then uncomment the following line
# no more 'woodchucks';

# version number
our $VERSION  = "1.5";

# global hash to hold command line options
our %opts     = ();

# global hash to hold terminal characteristics
our %terminal = ();

# number of still lives for a given population (OEIS A019473)
our @stilllives = (
    undef,	# 0
    0,		# 1
    0,		# 2
    0,		# 3
    2,		# 4
    1,		# 5
    5,		# 6
    4,		# 7
    9,		# 8
    10,		# 9
    25,		# 10
    46,		# 11
    121, 	# 12
    240,	# 13
    619,	# 14
    1353,	# 15
    3286,	# 16
    7773,	# 17
    19044,	# 18
    45759,	# 19
    112243,	# 20
    273188,	# 21
    672172,	# 22
    1646147,	# 23
    4051711,	# 24
);

# names of various object types
our %typenames = (
    "xs"		=> "still lifes",
    "xp"		=> "oscillators",
    "xq"		=> "spaceships",
    "yl"		=> "linear-growth patterns",
    "ov"		=> "oversized patterns",
    "zz"		=> "unclassified patterns",
    "PATHOLOGICAL"	=> "pathological patterns",
);

# names of the "p" (e.g. the 16 in xs16, the 4 in xq4 etc.) for the various
# types.
our %pnames = (
    "xs"		=> "population",
    "xp"		=> "period",
    "xq"		=> "period",
    "yl"		=> "period",	# ?
);

# standard spaceships
our %standard_spaceships = (
    "xq4_153"		=> "glider",
    "xq4_27dee6"	=> "MWSS",
    "xq4_27deee6"	=> "HWSS",
    "xq4_6frc"		=> "LWSS",
);

# well-known high-period (p>=5) oscillators
our %standard_oscillators = (
    "xp5_idiidiz01w1"		=> "octagon II",
    "xp6_ccb7w66z066"		=> "unix",
    "xp8_gk2gb3z11"		=> "figure-8",
    "xp8_g3jgz1ut"		=> "blocker",
    "xp8_wgovnz234z33"		=> "Tim Coe's p8",
    "xp14_j9d0d9j"		=> "tumbler",
    "xp15_4r4z4r4"		=> "pentadecathlon",
    "xp30_w33z8kqrqk8zzzx33"	=> "trans-queen-bee-shuttle",
    "xp30_w33z8kqrqk8zzzw33"	=> "cis-queen-bee-shuttle",
);

# well-known high-population (p>=30) still lifes
our %standard_stilllifes = (
    "xs30_69b8b96z69d1d96"	=> "?30-great sym",
    "xs30_caabaacz355d553"	=> "cis-mirrored_very_very_long_bee_siamese_eaters",
    "xs31_69b88bbgz69d11dd"	=> "Aries betwixt two blocks",
    "xs32_4a9b8b96z259d1d96"	=> "inflected 30-great-sym",
    "xs32_039u0u93z6a87078a6"	=> "cis-mirrored_worm_siamese_hook",
    "xs40_gj1u0u1jgzdlgf0fgld"	=> "omnibus",
);

# call main routine
exit MAIN();

### MAIN routine ###

sub MAIN {
    my $nicknames = {};		# nicknames hash
    my $block_fc  = 0;		# frequency class of block (xs4_33)
    my $logdata;		# data returned by read_logfiles
    my $starttime = time;	# when did we start?
    
    # automatically flush STDOUT.
    $OUTPUT_AUTOFLUSH = 1;
    
    # determine terminal size
    @terminal{
        "width-chars",
        "height-chars",
        "width-pixels",
        "height-pixels",
    } = GetTerminalSize();
    
    # parse command line options.
    parse_options();
    
    # help, version number or license requested?
    usage()   and return 1 if $opts{"help"};
    version() and return 1 if $opts{"version"};
    license() and return 1 if $opts{"license"};
    
    # if no files are specified, print usage information and return.
    usage()   and return 1 unless @ARGV;
    
    # read nicknames, if requested.
    $nicknames = read_nicknames("objectnames.txt")
        if $opts{"nickname-mode"};
    
    # read logfiles and extract returned data.
    $logdata   = read_logfiles(@ARGV);
    
    # if no logs were read, return.
    return 1 unless $logdata->{"stats"}->{"logs"};
    
    # frequency classes are computed related to the block's frequency.
    $block_fc  = log2($logdata->{"census"}->{"xs"}->{4}->{"xs4_33"});
    
    # ensure output directory/path exists.
    make_path $opts{"file-path"};

    # create graphs
    create_object_graphs($logdata);
    create_stats_graphs ($logdata);
    
#    say STDERR join "\n", grep { defined $_ && !is_weird_nickname($_) } map { $nicknames->{$_} } sort keys %{ $logdata->{'census'}->{'xs'}->{27} };
#    return 0;
    
    # write main css file
    output_css_file();
    
    # output index page (overview)
    output_index_file($logdata, $nicknames, $block_fc);
    
    # output unseen object page
    output_unseen_file($logdata, $nicknames, $block_fc);
    
    # output trophy page
    output_trophy_file($logdata, $nicknames, $block_fc);
    
    # output page for newly-seen objects    
    output_new_file($logdata, $nicknames, $block_fc);
    dump_seen_objects($logdata);
    
    # output diary
    output_diary($logdata, $nicknames);
    output_diary_stats_page($logdata);
    
    # output main census page (all objects).
    output_census_file($logdata, $nicknames, $block_fc);
    
    # output individual census pages for each object type.
    output_type_files($logdata, $nicknames, $block_fc);
    
    # create image files.
    create_images($logdata, $nicknames, $opts{"refresh-images"});

    say timestamped_msg(colored("All done", "green"), time - $starttime);

    # success!
    return 0;    
}

### options processing ###

# parse command-line options
sub parse_options {

    # default values.
    
    # cell size (in pixels) for images.
    $opts{"cellsize"}             = 4;
    
    # cell padding (in pixels) for images.
    $opts{"cellpadding"}          = 1;
    
    # nickname mode: 0=off, 1=on, 2=exclude weird.
    $opts{"nickname-mode"}        = 1;
    
    # force image refresh?
    $opts{"refresh-images"}       = 0;
    
    # graph width
    $opts{"graph-width"}          = 800;
    
    # graph height
    $opts{"graph-height"}         = 600;

    # number of common objects to show on index page
    $opts{"common-objects"}       = 100;
    
    # minimum oscillator period considered "high period"
    $opts{"high-period"}          = 5;

    # verbose output?
    $opts{"verbose"}              = 0;
    
    # when reading logs, indicate progress every ... files.
    $opts{"file-progress-step"}   = 100;
    
    # when reading logs, indicate progress every ... files.
    $opts{"object-progress-step"} = 1000;
    
    # interesting object code pattern
    $opts{"interesting-objects"}  =
        qr/
            ^						# anchor at the beginning
            (?:						# cluster, but don't capture
                 [^x]					# codes not starting with x are interesting
                |x[^s]					# codes not starting with xs are interesting
                |xs					# codes starting with xs MAY be interesting
                    ((?>\d+))				# capture population into $1, and don't backtrack
                    (??{ if(($1 < 26) && ($1 > 15)) {   # if 15 < population < 26...
                        '(?<!\w)' 			# ... interpolate a negative look-behind 
                    } })				# assertion guaranteed to fail.
                    _					# anchor in the middle of the code (literal _)
            )						# stop clustering
        /x;
    
    # catagolue URL.
    $opts{"catagolue-url"}        = "https://catagolue.appspot.com";
    
    # LifeWiki URL.
    $opts{"lifewiki-url"}         = "http://conwaylife.com/wiki";
    
    # file path.
    $opts{"file-path"}            = "census";
    
    # path to user's CSS file.
    $opts{"user-css"}             = undef;
    
    # output indent. The current value is an arbitrary constant, chosen so
    # that the longest current message will get three dots of padding.
    $opts{"output-indent"}        = 24;

    # parse options and store in %opts.
    # NOTE: GetOptions parses and removes ALL options from @ARGV, so it's
    # not possible to have more than one call to GetOptions.
    GetOptions(
        \%opts,
        "verbose|v+",
        "cell-size|cs=i",
        "cell-padding|cp=i",
        "graph-width|gw=i",
        "graph-height|gh=i",
        "common-objects|co=i",
        "nickname-mode|nm=i",
        "refresh-images|ri!",
        "catagolue-url|cu=s",
        "lifewiki-url|lu=s",
        "file-path|fp=s",
        "user-css|uc=s",
        "help|h",
        "version",
        "license",
    );
    
}

# print usage information
sub usage {
    my $cmd = basename $0;
    
    print <<USAGE;
$cmd $VERSION - aggregate statistics for apgnano log files

Usage: $cmd [options...] logfile [logfiles...]

Options:
        --cell-padding, -cp	Set cell padding (in pixels) for object images
        --cell-size, -cs	Set cell size (in pixels) for object images
        --nickname-mode,-nm	Set nickname mode:
                                    0 = off
                                    1 = on
                                    2 = exclude weird
        --refresh-images,-ri    Refresh object image files
        --graph-width,-gw       Set graph width (in pixels)
        --graph-height,-gh      Set graph height (in pixels)
        --verbose,-v		Enable verbose output (can be repeated)
        
        --catagolue-url,-cu	Set catagolue URL
        --lifewiki-url,-lu	Set LifeWiki URL
        --file-path,-fp		Set output path for files

        --help,-h               Display this message
        --license		Display license
        --version               Display version number
        
USAGE
}

# print version number
sub version {
    my $cmd = basename $0;
    
    say "$cmd $VERSION";
}

sub license {
    version();
    
    print <<LICENSE_END;
    
This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see [http://www.gnu.org/licenses/].
LICENSE_END
}

### Miscellaneous helper routines ###

# convert 0-9a-z to numbers 0-35.
sub ascii2num {
    my $n = shift;
    "0123456789abcdefghijklmnopqrstuvwxyz" =~ m/($n)/;
    return $LAST_MATCH_START[0];
}

sub log2 {
    my $n = shift;
    return log($n) / log(2);
}

# same as log2, but returns 0 for 0.
sub log2safe {
    my $n = shift;
    return 0 unless defined $n;
    return $n == 0 ? 0 : log2($n);
}

sub log10 {
    my $n = shift;
    return log($n) / log(10);
}

# same as log10, but returns 0 for 0.
sub log10safe {
    my $n = shift;
    return 0 unless defined $n;
    return $n == 0 ? 0 : log10($n);
}

# add commas to a number.
sub commify {
    my $n = reverse shift;
    $n =~ s/(\d{3})(?=\d)(?!\d*\.)/$1,/g;
    return scalar reverse $n;
}

# pad a message with a variable number of padding characters.
sub padded_msg {
    my ($msg, $char) = @_;
    return $msg . $char x ($opts{"output-indent"} - length($msg)) . " ";
}

# pad a message with dots
sub dotted_msg {
    return padded_msg(shift, ".");
}

# pad a message with spaces
sub spaced_msg {
    return padded_msg(shift, " ");
}

# print a message, in a given color, ensuring that if the terminal's width
# is reached, a new line with appropriate indenting is started.
sub indented_print {
    my ($msg, $color, $chars_written) = @_;
    
    # save old filehandle
    my $HANDLE = select *STDOUT;
    
    # if outputting this message would make us exceed the terminal width...
    if($chars_written + length($msg) >= $terminal{"width-chars"}) {
                
        # then start a new line...
        say   "";
                    
        # output spaces to vertically align the new line.
        print spaced_msg("");
                    
        # record the updated number of characters.
        $chars_written = $opts{"output-indent"} + 1;
    }
                
    # print the message
    print colored($msg, $color);
                
    # and record the number of chars.
    $chars_written += length $msg;
    
    # restore old filehandle
    select $HANDLE;
    
    # return how many characters have been written on the current line.
    return $chars_written;
}

# use singular or plural in a message, depending on what's appropriate
sub pluralify {
    my ($count, $singular, $plural, $rest) = @_;
    
    return $count . " " . (($count == 1) ? $singular : $plural) . $rest;
}

# add a timestamp (seconds elapsed) to a message.
sub timestamped_msg {
    my ($msg, $timestamp) = @_;
    
    $msg .= " in ";
    $msg .= sprintf("%.1f", $timestamp);
    $msg .= " seconds.";

    return $msg;
}

### apgcode-related routines ###

# apg uses letters w, x and y to create more compact codes. This routine
# decodes these and returns the "raw" code.
sub apg_decode_wxy {
    my $string = shift;

    # expand y0 (4 zeros) to yz (39 zeros).
    foreach my $char (0 .. 9, "a" .. "z") {
        $string =~ s/y$char/"0" x (ascii2num($char) + 4)/ge;
    }
    
    # expand w (2 zeros) and x (3 zeros).
    $string =~ s/w/00/g;
    $string =~ s/x/000/g;
    
    return $string;
}

# determine the population of an object. For oscillators, this is the
# population of the specific state encoded by the given code.
sub apg_population {
    my $apgcode = shift;
    
    # extract and prepare code.
    $apgcode =~ s/^.*_//;
    $apgcode =  apg_decode_wxy($apgcode);
    $apgcode =~ s/z//g;
    
    # count bits (live cells) and return the total.
    return sum0 map { hammingweight(ascii2num($_)) } split //, $apgcode;
}

### graph creation ###

# graph object data on a simulated log-scale graph. This contains most of
# the common code from create_object_graph and create_object_plot. 
# Unfortunately GD::Graph's API is baroque, to put it mildly.
sub create_graph {
    my ($data1, $type1, $data2, $type2, $x_min, $x_max, $x_label, $y_label, $title, $fname, $values_vertical) = @_;

    # image path
    my $pathname = "$opts{'file-path'}/images";
    
    # filename
    my $filename = "$pathname/$fname.png";
    
    # ensure image path exists    
    make_path $pathname;
    
    # log10-ify
    my @d1 = map { !(defined $_) || ($_ == 0) ? undef : (log10safe($_) + 1) } @$data1;

    # unfortunately we can't slice at the same time as taking a reference  
    @d1 = @d1[$x_min .. $x_max];

    # compile data for GD::Graph
    my @data = (
        [$x_min .. $x_max],
        \@d1,
    );

    # these are the values we'll print above each bar or point.    
    my @values = @data;
    
    # draw object distribution as a bar chart
    my @types = ($type1);
    
    # maximum Y axis value
    my $y_max = int(max(map { $_ // 0 } @d1)) + 1;

    if(defined $data2) {
        # log10-ify
        my @d2 = map { log10safe($_) + 1 } @$data2;
    
        # again, slice
        @d2 = @d2[$x_min .. $x_max];
        
        # add second data set to data array
        push @data, \@d2;

        # values for the second data set should not be drawn, so we provide
        # the appropriate number of undefs
        push @values, [(undef) x (scalar @d1)];
        
        # adjust Y axis maximum value, if necessary
        $y_max = max(
            $y_max, 
            int(max(map { $_ // 0 } @d2)) + 1
        );
        
        # remember what type of graph (line, bar, point, ...) to use for the
        # second data set
        push @types, $type2;
    }
    
    # create a new GD::Graph
    my $graph = GD::Graph::mixed->new($opts{"graph-width"}, $opts{"graph-height"});

    # helper sub to format numbers; this is to simulate a log scale.
    my $number_formatter = sub {
        my $label = shift;
        return $label 
            ? commify(int(10 ** ($label - 1) + 0.5))
            : 0;
        };

    # set graph options    
    $graph->set(
        # general options
        types			=> \@types,
        line_width		=> 5,
        transparent		=> 0,
        
        # text labels
        x_label			=> $x_label,
        y_label			=> $y_label,
        title			=> $title,

        # y-axis
        y_tick_number		=> $y_max,
        y_max_value		=> $y_max,
        y_number_format		=> $number_formatter,

        # x-axis
        x_label_position	=> 1/2,

        # values in the graph
        show_values		=> \@values,
        skip_undef		=> 1,
        hide_overlapping_values	=> 1,
        values_vertical		=> $values_vertical,
        values_format		=> $number_formatter,
        
    ) or die $graph->error;
    
    # plot graph
    my $gd = $graph->plot(\@data)
        or die $graph->error;

    # write file        
    open my $FILE, ">", $filename
        or die "Cannot open $filename for writing: $!\n";
    
    binmode $FILE;
    print $FILE $gd->png;
    
    close $FILE
        or warn "Cannot close $filename: $!\n";

}

# create a bar/line graph showing the number of distinct objects seen for
# each p, plus number of distinct possible objects (if known).
sub create_object_graph {
    my ($logdata, $type, $p_name, $possible_objects, $chars_written) = @_;

    # p's seen
    my @ps = sort { $a <=> $b } keys %{ $logdata->{"census"}->{$type} };

    # minimum and maximum p seen    
    my $min_p = min @ps;
    my $max_p = max @ps;

    # collect number of distinct objects of each p
    my @num_objects = ();
    foreach my $p (@ps) {
        $num_objects[$p] = scalar keys %{ $logdata->{"census"}->{$type}->{$p} };
    }
    
    # initialize missing elements to zero
#    $_ //= 0 foreach (@num_objects);

    create_graph(
        \@num_objects,				# data1
        "bars",					# type1
        $possible_objects,			# data2
        "lines",				# type2
        $min_p,					# x_min
        $max_p,					# x_max
        ucfirst $p_name,			# x_label
        "Distinct # $typenames{$type} seen",	# y_label
        "Distinct $typenames{$type}",		# title
        $type,					# filename (.png)
        1					# values_vertical
    );
    
    # if in verbose mode, output object type.
    if($opts{"verbose"}) {
        $chars_written = indented_print("$type ", "bright_blue", $chars_written);
    }
    
    return $chars_written;
    
}

# create a point graph showing the total number of objects seen for each p
# (population/period).
sub create_object_plot {
    my ($logdata, $type, $p_name, $chars_written) = @_;
    
    # ps seen
    my @ps = sort { $a <=> $b } keys %{ $logdata->{"census"}->{$type} };

    # minimum and maximum p seen    
    my $min_p = min @ps;
    my $max_p = max @ps;

    # collect total number of objects of each p
    my @num_objects = ();
    foreach my $p (@ps) {
        $num_objects[$p] = sum0 values %{ $logdata->{"census"}->{$type}->{$p} };
    }
    
    create_graph(
        \@num_objects,				# data1
        "points",				# type1
        undef,					# data2
        undef,					# type2
        $min_p,					# x_min
        $max_p,					# x_max
        ucfirst $p_name,			# x_label
        "Total # $typenames{$type} seen",	# y_label
        "Total $typenames{$type}",		# title
        "${type}_total",			# filename (.png)
        0					# values_vertical
    );

    # if in verbose mode, output object type.
    if($opts{"verbose"}) {
        $chars_written = indented_print("${type}_total ", "bright_blue", $chars_written);
    }
    
    return $chars_written;
    
}

# create object graphs
sub create_object_graphs {

    my ($logdata) = @_;
    my $starttime = time;

    # number of characters written to terminal (used for line breaks).  to
    # start, the equivalent of one dotted_msg.
    my $chars_written = $opts{"output-indent"} + 1;
    
    print dotted_msg("Creating object graphs");

    $chars_written = create_object_graph($logdata, "xs", "population", \@stilllives, $chars_written);
    $chars_written = create_object_graph($logdata, "xp", "period",     undef,        $chars_written);
    $chars_written = create_object_plot ($logdata, "xs", "population",               $chars_written);
    $chars_written = create_object_plot ($logdata, "xp", "period",                   $chars_written);

    # if in verbose mode, and if files were written, output a new line.
    if($opts{"verbose"}) {
        say   "";
        print spaced_msg("");
    }

    say timestamped_msg(
        colored(
            pluralify(4, "graph", "graphs", " created"), 
            "green"), 
        time - $starttime
    );
}

# create a line graph showing some daily diary stat.
sub create_daily_stats_graph {
    my ($logdata, $stat, $title, $chars_written) = @_;
    
    # flatten stats and set undefs to 0.
    my %flatstats = map {
            ($_, $logdata->{'diary'}->{$_}->{'stats'}->{$stat} // 0)
        } keys %{ $logdata->{'diary'} };

#    use Data::Dumper;
#    say Dumper \%flatstats;
    
    my @statskeys   = sort keys %flatstats;
    my @statsvalues = map { $flatstats{$_} } @statskeys;

    # image path
    my $pathname = "$opts{'file-path'}/images";
    
    # filename
    my $filename = "$pathname/daily${stat}.png";
    
    # ensure image path exists    
    make_path $pathname;
    
    # compile data for GD::Graph
    my @data = (
        \@statskeys,
        \@statsvalues
    );
    
    # draw object distribution as a bar chart
    my @types = ("lines");
    
    # maximum Y axis value
    my $y_max = int(max(map { $_ // 0 } @statsvalues)) + 1;

    # create a new GD::Graph
    my $graph = GD::Graph::mixed->new($opts{"graph-width"}, $opts{"graph-height"});

    # set graph options    
    $graph->set(
        # general options
        types			=> \@types,
        line_width		=> 5,
        transparent		=> 0,
        
        # text labels
        x_label			=> "Date",
        y_label			=> $title,
        title			=> $title,

        # x-axis
        x_label_position	=> 1/2,
        x_labels_vertical	=> 1,
        x_label_skip            => 7,
    ) or die $graph->error;
    
    # plot graph
    my $gd = $graph->plot(\@data)
        or die $graph->error;

    # write file        
    open my $FILE, ">", $filename
        or die "Cannot open $filename for writing: $!\n";
    
    binmode $FILE;
    print $FILE $gd->png;
    
    close $FILE
        or warn "Cannot close $filename: $!\n";

    # if in verbose mode, output object type.
    if($opts{"verbose"}) {
        $chars_written = indented_print("daily${stat} ", "bright_blue", $chars_written);
    }
    
    return $chars_written;
    
}

sub create_stats_graphs {
    my ($logdata) = @_;
    my $starttime = time;

    # number of characters written to terminal (used for line breaks).  to
    # start, the equivalent of one dotted_msg.
    my $chars_written = $opts{"output-indent"} + 1;
    
    print dotted_msg("Creating stats graphs");
    
    $chars_written = create_daily_stats_graph($logdata, "logs",           "Hauls",            $chars_written);
    $chars_written = create_daily_stats_graph($logdata, "soups",          "Soups",            $chars_written);
    $chars_written = create_daily_stats_graph($logdata, "soupsperhaul",   "Soups per haul",   $chars_written);
    $chars_written = create_daily_stats_graph($logdata, "objects",        "Total objects",    $chars_written);
    $chars_written = create_daily_stats_graph($logdata, "distinct",       "Distinct objects", $chars_written);
    $chars_written = create_daily_stats_graph($logdata, "new",            "New objects",      $chars_written);
    $chars_written = create_daily_stats_graph($logdata, "newxps",         "New oscillators",  $chars_written);
    $chars_written = create_daily_stats_graph($logdata, "newxss",         "New still lifes",  $chars_written);
    $chars_written = create_daily_stats_graph($logdata, "newxqs",         "New spaceships",   $chars_written);
    $chars_written = create_daily_stats_graph($logdata, "newyls",         "New linear-growth patterns", $chars_written);
    $chars_written = create_daily_stats_graph($logdata, "objectspersoup", "Objects per soup", $chars_written);

    foreach my $type (keys %{ $logdata->{'census'} }) {
        foreach my $p (keys %{ $logdata->{'census'}->{$type} }) {
            $chars_written = create_daily_stats_graph($logdata, "new${type}${p}s", "New ${type}${p}s", $chars_written);
        }
    }

    # if in verbose mode, and if files were written, output a new line.
    if($opts{"verbose"}) {
        say   "";
        print spaced_msg("");
    }

    say timestamped_msg(
        colored(
            pluralify(7, "graph", "graphs", " created"), 
            "green"), 
        time - $starttime
    );

}

### object image creation

# get image file path, relative to HTML file path.
sub image_path {
    return "images/" . $opts{"cellsize"} . "." . $opts{"cellpadding"};
}

# create image files for objects in log data and/or nicknames
sub create_images {
    my ($logdata, $nicknames, $force_refresh) = @_;
    
    my @log_objects      = keys %{ $logdata->{'flat'} };
    my @nickname_objects = keys %$nicknames;
    
    my @all_objects      = get_union([ \@log_objects, \@nickname_objects ]);
    
    create_all_images(\@all_objects, $force_refresh);
}    

# create all image files
sub create_all_images {
    # FIXME: should force_refresh really be a parameter here, or
    # should create_image_file read from %opts instead?
    my ($objects, $force_refresh) = @_;
    
    # starting time
    my $starttime     = time;

    # number of new image files written
    my $images_created = 0;
    
    # number of characters written to terminal (used for line breaks).  to
    # start, the equivalent of one dotted_msg.
    my $chars_written = $opts{"output-indent"} + 1;
    
    print dotted_msg("Creating images");

    # iterate through object census hash    
    foreach my $object (@$objects) {
    
        # create image file and count it if it was indeed created.
        if(create_image_file($object, $force_refresh)) {
        
            # if in verbose mode, output object type.
            if($opts{"verbose"}) {
                $chars_written = indented_print(
                    "$object ", 
#                    ($object =~ $opts{"interesting-objects"} ? "bright_blue" : "blue"),
                    "bright_blue",
                    $chars_written
                );
            }
            
            # record the fact an image file was created.
            $images_created++;
        }
        
    }
    
    # if in verbose mode, and if any image names were output, start a new line.
    if($opts{"verbose"} && $images_created) {
        say   "";
        print spaced_msg("");
    }
    
    say timestamped_msg(
        colored(
            pluralify(
                $images_created,
                "image", 
                "images", 
                " created"
            ), 
            "green"), 
        time - $starttime
    );
    
}

# create an image file for a given object, if none exists yet.
# returns 1 if an image was created, 0 otherwise.
sub create_image_file {
    my ($object, $force_refresh) = @_;
    my $apgcode;
    
    my $pathname = "$opts{'file-path'}/" . image_path();
    my $filename = "$pathname/$object.png";

    # skip image if it exists
    return 0 if -e $filename && !$force_refresh;
    
    # skip image if it's a linear-growth pattern (yl), oversized (ov), weird
    # (zz) or something else that's not a still life, spaceship or
    # oscillator
    return 0 if $object !~ m/^x/;

    # ensure image path exists    
    make_path $pathname;

    # extract/decode apg code into something useful: strip object type
    # identifier, then decode w, x and y codes.
    $apgcode =  $object;
    $apgcode =~ s/^.*_//;
    $apgcode =  apg_decode_wxy($apgcode);
    
    # split code into strips. These will be rendered individually.            
    my @strips  = split /z/, $apgcode;
    
    # determine highest bit set in the last strip.  This is so we won't have
    # unnecessary empty rows at the end of the image.
    my $highest_bit =
        max
        map { int(log2safe(ascii2num($_))) }
        split //, $strips[-1];
        
    # determine image width and height (in cells, not pixels), and total
    # cell size (including padding).
    my $width         = max map { length } @strips;
    my $height        = (scalar @strips) * 5 - (4 - $highest_bit);
    my $totalcellsize = $opts{"cellsize"} + 2 * $opts{"cellpadding"};
    
    # create GD image, allocate colors
    my $image = new GD::Image($width  * $totalcellsize, $height * $totalcellsize);
    my $white = $image->colorAllocate(255, 255, 255);
    my $black = $image->colorAllocate(  0,   0,   0);
    
    # make background transparent.
    $image->transparent($white);

    # image is rendered in strips of five rows each.
    foreach my $currentstrip (0 .. $#strips) {
    
        # for each strip, every character encodes a single column of five
        # rows.
        my @chars = split //, $strips[$currentstrip];
        foreach my $currentchar (0 .. $#chars) {
        
            # each bit in the character determines whether the corresponding
            # cell is live or not.
            my $number = ascii2num($chars[$currentchar]);
            for my $bit (0 .. 4) {

                # if the cell is live, draw a rectangle at the right
                # position, taking into account cell size and padding.
                if($number & (2 ** $bit)) {
                    my $x1 = $currentchar * $totalcellsize + $opts{"cellpadding"};
                    my $y1 = ($currentstrip * $totalcellsize * 5) + ($bit * $totalcellsize) + $opts{"cellpadding"};

                    $image->filledRectangle($x1, $y1, $x1 + $opts{"cellsize"} - 1, $y1 + $opts{"cellsize"} - 1, $black);
                }
                
            }
        }
    }

    # now that the image is created, open a file...
    open my $IMAGEFILE, ">", $filename
        or do {
            warn "Cannot open $filename for writing: $!\n";
            return 0;
        };
    
    # ...write the image to it...
    binmode $IMAGEFILE;
    print $IMAGEFILE $image->png;
    
    # ...and close it.
    close $IMAGEFILE
        or warn "Cannot close $filename: $!\n";
        
    # 1 new image file created!
    return 1;
}

### logfile routines ###

# read a number of logfiles; return an object census hash, flat object
# census hash, and statistics hash, all wrapped up in another hashref.
sub read_logfiles {

    my $objects     = {};		# object census hashref
    my $flatobjects = {};		# flat object census hashref
    my $diary       = {};		# diary of interesting events
    my $starttime   = time;		# time we started reading logs

    my $stats       = { 		# statistics gathered
        "logs"	=> 0,
        "time"	=> DateTime->now(time_zone => 'local'),
    };
    
    print dotted_msg("Reading logfiles");
    
    # characters written to current output line
    my $chars_written = $opts{'output-indent'} + 1;
    
    # iterate over filenames    
    LOGFILE: foreach my $logfile (@_) {  
    
        # open logfile, skip if unreadable
        open my $LOGFILE, "<", $logfile
          or do {
            warn "Cannot open $logfile for reading: $!\n";
            next LOGFILE;
        };
        
        my $timestamp = undef;
        
        # does the filename match the standard format? If not, the file was
        # renamed, and we refuse to extract the log's timestamp from the
        # name.
        if($logfile =~ m#(?:^|/)log\.(\d+)\.(?:(?:n|m)_)?.{12}\.txt$#) {
        
            my ($day, $mon, $year) = (localtime $1)[3 .. 5];
            $timestamp = sprintf "%04d-%02d-%02d", $year + 1900, $mon + 1, $day;
            
            # this ensures that we'll get a diary page even if there's no
            # new objects on a given day.
            $diary->{$timestamp}->{'newobjects'} //= {};
            
            $diary->{$timestamp}->{'stats'}->{'logs'}++;
            $stats->{'diarylogs'}++;
        } else {
            warn "Non-standard logfile name: $logfile";
        }

        # count (readable) log files
        $stats->{'logs'}++;
        
        $chars_written = indented_print("$stats->{'logs'} ", "bright_blue", $chars_written)
            if ($stats->{'logs'} % $opts{'file-progress-step'} == 0)
                && $opts{'verbose'};
            
        # munge logfile
        while(<$LOGFILE>) {
        
            # make extra-sure there's no stray CRs or LFs
            chomp;
            s/\r$//;
            
            # extract number of soups in log file
            m/^\@NUM_SOUPS (\d+)$/   and do {
                $stats->{'soups'} += $1;
                
                if(defined $timestamp) {
                    $diary->{$timestamp}->{'stats'}->{'soups'} += $1;
                }
            }; 
            
            # extract number of objects in log file
            m/^\@NUM_OBJECTS (\d+)$/ and do {
                $stats->{'objects'} += $1;
                
                if(defined $timestamp) {
                    $diary->{$timestamp}->{'stats'}->{'objects'} += $1;
                }
            };
            
            # read only from "@CENSUS TABLE" on, and up to either "@TOP 100" or
            # "@SAMPLE_SOUPIDS" (or end of file).
            # FIXME: this probably makes too many assumptions about the log
            # file format.
            next unless m/^\@CENSUS TABLE$/ .. m/^\@(?:TOP 100|SAMPLE_SOUPIDS)$/ || eof;
            
            # skip lines that start with an "@" as well as empty lines.
            next if     m/^\@/;
            next unless length;
    
            # extract and save relevant information. "p" can stand for
            # either "population" or "period".
            # FIXME: Could use named captures instead?
            m/^((\w{2})(\d+)_[^ ]+) (\d+)$/;
            my ($object, $type, $p, $count) = ($1, $2, $3, $4);
            
            my $prefix = $type . $p;
            
            # sanity check: if we couldn't parse an object on this line,
            # something's wrong.
            next unless defined $object;
            
            # if we know the timestamp of this log, we may record
            # interesting events in the diary.
            if(defined $timestamp) {
            
                $diary->{$timestamp}->{'census'}->{$type}->{$p}->{$object} += $count;
                $diary->{$timestamp}->{'flat'}                 ->{$object} += $count;
                
                # was the object seen before? If not, record its first
                # occurence.
                unless($flatobjects->{$object}) {
                    push @{ $diary->{$timestamp}->{'newobjects'}->{$type}->{$p} }, $object;
                    $diary->{$timestamp}->{'stats'}->{'new'}++;
                    $diary->{$timestamp}->{'stats'}->{"new${type}s"}++;
                    $diary->{$timestamp}->{'stats'}->{"new${prefix}s"}++;
                }
                
            }
            
            # record object/count in both census hash and flat census hash.
            $objects->{$type}->{$p}->{$object} += $count;
            $flatobjects           ->{$object} += $count;
    
        }
        
        # close logfile
        close $LOGFILE
            or warn "Cannot close $logfile: $!\n";
            
    }

    # determine number of distinct, new etc. objects    
#    $stats->{'distinct'} = sum0 map { scalar keys %$_ } map { values %$_ } values %$objects;
    $stats->{'distinct'} = scalar keys %$flatobjects;
    
    # this is set primarily to mirror the diary stat, and avoid unpleasant
    # surprises down the road if the key doesn't exist.  The stat isn't
    # otherwise meaningful.
    $stats->{'new'}      = scalar keys %$flatobjects;

    $stats->{'soupsperhaul'}   = int             ($stats->{'soups'}   / $stats->{'logs'} );
    $stats->{'objectspersoup'} = sprintf "%.1f", ($stats->{'objects'} / $stats->{'soups'});

    foreach my $date (keys %$diary) {
        $diary->{$date}->{'stats'}->{'distinct'}       = scalar keys %{ $diary->{$date}->{'flat'}       };
        
        $diary->{$date}->{'stats'}->{'soupsperhaul'}   = int             ($diary->{$date}->{'stats'}->{'soups'}   / $diary->{$date}->{'stats'}->{'logs'} );
        $diary->{$date}->{'stats'}->{'objectspersoup'} = sprintf "%.1f", ($diary->{$date}->{'stats'}->{'objects'} / $diary->{$date}->{'stats'}->{'soups'});
    }
    
    # first and last day for which there is diary data, as Date::Simple
    # objects.
    my ($firstday, $lastday) = 
        map {
            Date::Simple->new($_)
        } (sort keys %$diary)[0, -1];
    
    # date range of days for which there is diary data    
    my $diarydays = Date::Range->new($firstday, $lastday);

    # initialize missing diary date stats to empty hashes (i.e. hashrefs).
    for my $date ($diarydays->dates()) {
        $diary->{$date}->{'stats'} //= {};
    }
    
    # if we output our progress above while reading, start a new line.
    if($opts{'verbose'} && $stats->{'logs'} >= $opts{'file-progress-step'}) {
        say   "";
        print spaced_msg("");
    }
        
    say timestamped_msg(
        colored(
            pluralify(
                $stats->{'logs'}, 
                "log file", 
                "log files", 
                " read"), 
            "green"), 
        time - $starttime
    );

    # return data gathered, as an anonymous hashref
    return {
        "census" => $objects, 
        "flat"   => $flatobjects,
        "stats"	 => $stats,
        "diary"  => $diary,
    };
}

### nickname routines ###

sub make_object_link {
    my ($object, $nicknames, $indicate_interesting) = @_;
    
    my $ret = "";
    
    # if object is interesting, wrap everything in the right CSS class.
    if($indicate_interesting && ($object =~ $opts{'interesting-objects'})) {
        $ret .= qq#<span class="interesting">#;
    }

    # add link to catagolue page for this object
    $ret .= qq#<a href="$opts{"catagolue-url"}/object/$object/b3s23">$object</a>#;

    # if there is a nickname, add that. If the nickname is not weird, also
    # make it a link to whatever LifeWiki article may exist for this object.
    if(exists $nicknames->{$object}) {	       
        if(is_weird_nickname($nicknames->{$object})) {
            $ret .= " ($nicknames->{$object})";
        } else {
            $ret .= qq# (<a href="$opts{"lifewiki-url"}/$nicknames->{$object}">$nicknames->{$object}</a>)#;
        }
    }
    
    # if we opened a span, we gotta close it again
    if($indicate_interesting && ($object =~ $opts{'interesting-objects'})) {
        $ret .= '</span>';
    }
    
    return $ret;
}

# read nicknames from a given file.
sub read_nicknames {
    my $filename  = shift;
    my $starttime = time;

    # nickname hashref
    my $nicknames = {};

    # make sure file exists...
    unless(-e $filename) {
        warn "$filename not found.\n";
        return $nicknames;
    }

    # ... and is readable.
    open my $NAMESFILE, "<", $filename
      or do {
        warn "Cannot open $filename for reading: $!\n";
        return $nicknames;
    };

    print dotted_msg("Reading nicknames");

    # munge file
    my $nicknames_read = 0;
    NICKNAME: while(<$NAMESFILE>) {
        chomp;
        
        # file format is very simple: object code, space, nickname (possibly
        # including more spaces).
        my ($object, $nickname) = split / /, $_, 2;
        
        # "weird" nicknames, for the moment, are those that have square
        # brackets, question marks or underscores in them.  Depending on our
        # mode, we skip these.
        next NICKNAME if is_weird_nickname($nickname) && $opts{"nickname-mode"} == 2;

        # record and count nickname.
        $nicknames->{$object} = $nickname;
        $nicknames_read++;
        
    }
    
    # close file
    close $NAMESFILE
        or warn "Cannot close $filename: $!\n";
            
    say timestamped_msg(colored(pluralify($nicknames_read, "nickname", "nicknames", " read"), "green"), time - $starttime);
        
    return $nicknames;
}

# determine whether a given nickname is "weird".
sub is_weird_nickname {
    my $nickname = shift;
    
    # nickname is "weird" if it contains [, ], ? or _ characters (others may
    # be added later).
    return $nickname =~ m/[\[\]\?_]/;
}

### HTML creation routines ###

# format totals as a string
sub totals {
    my ($dataref) = @_;
    my $ret = "";
    
    $ret .= commify ($dataref->{"stats"}->{'logs'}          ) . " hauls, ";
    $ret .= commify ($dataref->{"stats"}->{'soups'}         ) . " soups (&#x2300; ";
    $ret .= commify ($dataref->{"stats"}->{'soupsperhaul'}  ) . " per haul), ";
    $ret .= commify ($dataref->{"stats"}->{'objects'}       ) . " objects (";
    $ret .= commify ($dataref->{"stats"}->{'distinct'}      ) . " distinct), &#x2300; ";
    $ret .=          $dataref->{"stats"}->{'objectspersoup'}  . " objects per soup.";
    
    return $ret;
}

# output a HTML table tabulating a number of objects: image, code/nickname,
# count, frequency class relative to a given base frequency class
# (computed), and frequency class relative to the group's most common
# objects (computed).
sub output_object_table {
    my ($FILE, $title, $HTMLsnippet, $objects, $nicknames, $base_fc, $possible_objects, $progress, $total_object_count, $maximum_output_lines, $columns, $indicate_interesting) = @_;
    
    # we're doing a lot of outputting to the same handle, so to save typing,
    # select() that handle (and save the old one so we can restore it).
    my $OLD_FILEHANDLE = select $FILE;
    
    # total number of objects in group
    my $group_total     = sum0 values %$objects;
    
    # output h1 title, if desired
    if(defined $title) {
        print "<h1>$title (" . commify(scalar (keys %$objects)) . " distinct";
    
        if(defined $possible_objects) {
            print ", out of " . commify($possible_objects);
        }
    
        print "; " . commify($group_total) . " total";
    
        say ")</h1>";
    }
    
    say $HTMLsnippet;
    
    # output table header
    say '<table border="1" class="objecttable">';
    say "<tr>";
    say "<th>Rank</th>";
    say "<th>Object</th>";
    say "<th>Population</th>";
    say "<th>Code</th>";
    
    if($columns > 0) {
        say "<th>Count</th>";
        say "<th>FC</th>";
        say "<th>FC (group)</th>"  if $columns > 1;
        say "<th>1/f</th>";
        say "<th>1/f (group)</th>" if $columns > 1;
    }
    
    say "</tr>";
        
    my $group_fc        = undef;	# base frequency class for this group
    my $rank            = 0;		# rank of current object
    my $rank_skipped    = 0;		# ranks skipped due to objects being tied at the same rank
    my $prev_count      = -1;		# previous object's count
    my $objects_written = 0;		# number of distinct objects written

    # characters written to terminal on current line.
    # FIXME: this REALLY doesn't belong here.
    my $chars_written = $opts{'output-indent'} + 1;
    
    # iterate over all objects, from most common to least common. If objects
    # were seen equally often, they appear in lexicograph order.
    foreach my $object (sort { $objects->{$b} <=> $objects->{$a} || $a cmp $b } keys %$objects) {

        # how often was this object seen?
        my $count             = $objects->{$object};
        
        # determine object's population
        my $population        = apg_population($object);

        # determine object's absolute frequency class
        my $absolute_fc       = log2safe($count);

        # use the first object (i.e. the most commonly seen one) as
        # reference point for the group frequency class.  Note that this
        # will break if the sort order is changed in the foreach above.
        $group_fc //= $absolute_fc;			

        # determine object's frequency class relative to block and group leader.           
        my $relative_fc       = ($base_fc - $absolute_fc);
        my $group_relative_fc = ($group_fc - $absolute_fc);
        
        # determine if object is tied with previous object, and adjust rank
        # accordingly.
        if($count == $prev_count) {
            $rank_skipped += 1;
            
        } else {
            $rank         += $rank_skipped + 1;
            $rank_skipped  = 0;
            
        }
        
        # if the number of lines to be output is limited, and we're over it
        # now, stop outputting lines (including this one).
        last if(defined $maximum_output_lines && ($rank > $maximum_output_lines));
        
        # now, finally output a table row.
        say "<tr>";

        # output rank.
        print '<td class="number">';
        
        if($rank_skipped) {
            print "=$rank (" . ($rank + $rank_skipped) . ")";
        } else {
            print $rank;
        }
        
        say "</td>";
        
        # output image.
        if($object =~ m/^x/) {
            say qq#<td class="oimg"><img src="# . image_path() . qq#/$object.png" /></td>#;
        } else {
            say qq#<td></td>#;
        }
            
        # output population. It's probably unnecessary to use commify here.
        say '<td class="number">' . commify($population) . "</td>";

        # output object code and nickname
        say "<td>" . make_object_link($object, $nicknames, $indicate_interesting) . "</td>";
        
        
        
        if($columns > 0) {
            # output count.
            say '<td class="number">' . commify($count) . "</td>";
            
            # output frequency class.
            print  '<td class="number">';
            printf "%.1f", $relative_fc;
            say    "</td>";
            
            # output group frequency class.
            if($columns > 1) {
                print  '<td class="number">';
                printf "%.1f", $group_relative_fc;
                say    "</td>";
            }
            
            # output 1/f.
            print  '<td class="number">';
            printf "%6.6g", ($total_object_count / $count);
            say    "</td>";
            
            # output group 1/f.
            if($columns > 1) {
                print  '<td class="number">';
                printf "%6.6g", ($group_total / $count);
                say    "</td>";
            }
        }
            
        # end table row.
        say "</tr>";
        
        # remember object count so we can determine whether next object is
        # tied for rank.
        $prev_count = $count;
        
        $chars_written = indented_print("$objects_written ", "bright_blue", $chars_written)
            if($progress && !(++$objects_written % $opts{'object-progress-step'}));
            
    }
    
    # end table.
    say "</table>";
    
    # restore previously-selected filehandle.
    select $OLD_FILEHANDLE;
}

# start a HTML page and output all the necessary basics
sub output_css_file {

    # open file
    open my $CSSFILE, ">", "$opts{'file-path'}/census.css"
        or die "Cannot open census.css for writing: $!\n";
        
    # save currently selected filehandle, and select output filehandle
    my $OLD_FILEHANDLE = select $CSSFILE;

    say <<ENDOFCSS;
/* NOTE: This file is automatically-generated.
 * CHANGES MADE WILL BE OVERWRITTEN.
 */

html {
    background:			black;
    background-image:		url(images/bg.png);
    background-attachment:	fixed;
    color:			white;
}

.contentcontainer {
    width:			90%;
    margin-left:		auto;
    margin-right:		auto;
    
    padding:			10px;
    border-radius:		10px;
    background:			rgba(255, 255, 255, 0.9);
    color:			black;
}

table.objecttable {
    width:			90%;
    margin-left:		auto;
    margin-right:		auto;
    
    border:			1px solid black;
    border-collapse:		collapse;
}

table.objecttable td {
    padding:			5px;
}

.number {
    text-align:			right;
}

.oimg {
    text-align:			center;
}

.graph {
    display:			block;
    margin-left:		auto;
    margin-right:		auto;
    margin-bottom:		1em;
    width:			$opts{"graph-width"};
}

.graph-container {
    width:			100%;
}

.trophycase {
    margin-left:		auto;
    margin-right:		auto;
    width:			50%;
    text-align:			center;
}

.interesting {
    font-weight:		bold;
}

ENDOFCSS

    # restore previously-selected filehandle
    select $OLD_FILEHANDLE;
    
    # close file
    close $CSSFILE
        or warn "Cannot close census.css: $!\n";

}

# start a HTML page and output all the necessary basics
sub output_header {
    my ($FILE, $logdata) = @_;

    # save currently selected filehandle, and select output filehandle to
    # save typing
    my $OLD_FILEHANDLE = select $FILE;

    # output HTML header etc.
    say "<!DOCTYPE html>";
    say "<html>";
    say "<head>";
    say qq#<link rel="stylesheet" type="text/css" href="census.css" />#;
    
    if(defined $opts{"user-css"}) {
        say qq#<link rel="stylesheet" type="text/css" href="$opts{"user-css"}" />#;
    }
    
    say "</head>";
    say "<body>";
    say '<div class="contentcontainer">';
    
    # output overall statistics
    print  "<p>";
    print "Stats generated " . $logdata->{"stats"}->{'time'}->ymd . " " . $logdata->{"stats"}->{'time'}->hms . "; totals: ";
    say   totals($logdata) . "</p>";

    # restore previously-selected filehandle
    select $OLD_FILEHANDLE;
}

# wrap up and end a HTML page
sub output_footer {
    my $FILE = shift;

    # save currently selected filehandle, and select output filehandle to
    # save typing
    my $OLD_FILEHANDLE = select $FILE;

    # output an explanatory note regarding frequency classes    
    say "<p>FC (frequency class): reference object (block or most common object in group) is 2<sup>FC</sup> times as common as this object. 1/f: one in (1/f) objects encountered is of this type.</p>";
    
    # output an explanatory note regarding linear-growth patterns
    say "<p>Note: linear-growth patterns (yl_*) may be severely underrepresented.</p>";

    # output HTML footer
    say "</div>";
    say "</body>";
    say "</html>";

    # restore previously-selected filehandle
    select $OLD_FILEHANDLE;
}

# output navigation links
sub output_navigation {
    my ($FILE, $logdata) = @_;
    
    # save currently selected filehandle, and select output filehandle to
    # save typing
    my $OLD_FILEHANDLE = select $FILE;

    # we're presenting these as a list of lists for the time being.
    say "<ul>";
    
    say '<li>All: <a href="index.html">index</a> &bull; ';
    say '<a href="census.html">census</a> (warning: huge page) &bull; ';
    say '<a href="new.html">new objects</a> &bull; ';
    say '<a href="trophy.html">trophy case</a> &bull; ';
    say '<a href="dailystats.html">daily stats</a> &bull; ';
    say '<a href="unseen.html">unseen objects</a></li>';
    
    say '<li>Objects:';
    say '<ul>';
    
    # iterate over basic types (xs, xq, xp etc.) and output a list item and
    # sublist for each.
    foreach my $type (sort keys %{ $logdata->{"census"} }) {
    
        say "<li>" . ucfirst($typenames{$type}) . ": ";
        
        # populate sublist with links for type/p combinations.
        say
            join  " &bull; ", 
            map   { qq#<a href="$type$_.html">$type$_</a># }
            sort  { $a <=> $b } 
            keys %{ $logdata->{"census"}->{$type} }
        ;
        
        say "</li>";
    }
    
    say '</ul>';
    say '</li>';
    
    say '<li>Diary:';
    say '<ul>';

    # skip days for which no data was submitted
    my @diarydates = grep { exists $logdata->{'diary'}->{$_}->{'census'} } keys %{ $logdata->{'diary'} };
        
    foreach my $month (uniq sort { $a cmp $b } map { substr $_, 0, 7 } @diarydates) {
        say "<li>$month: ";
        
        say
            join " &bull; ",
            map  { qq#<a href="diary$month-$_.html">-$_</a># }
            sort { $a <=> $b }
            map  { substr $_, -2, 2 } 
            grep { m/^$month/ } 
                 @diarydates
        ;
        
        say '</li>';
    
    }
    
    say '</ul>';
    say '</li>';
    
#    say '<li>Graphs: <a href="images/xs.png">distinct still lifes</a> &bull; <a href="images/xp.png">distinct oscillators</a> &bull; <a href="images/xs_total.png">total still lifes</a> &bull; <a href="images/xp_total.png">total oscillators</a></li>';

    # wrap up list
    say "</ul>";
    
    # restore previously-selected filehandle
    select $OLD_FILEHANDLE;
}

# output a complete HTML page with a census for a given bunch of objects.
sub output_census_page {
    my ($filename, $title, $HTMLsnippet, $logdata, $objects, $nicknames, $base_fc, $total, $progress, $columns, $indicate_interesting) = @_;
    
    # open file
    open my $CENSUSFILE, ">", "$opts{'file-path'}/$filename"
        or die "Cannot open $filename for writing: $!\n";
    
    # output page header
    output_header(
        $CENSUSFILE,	# filehandle
        $logdata,	# log data
    );
    
    # output navigation links
    output_navigation(
        $CENSUSFILE,	# filehandle
        $logdata,	# log data
    );
    
    # output census table
    output_object_table(
        $CENSUSFILE,    # filehandle
        $title,         # title
        $HTMLsnippet,	# HTML snippet
        $objects,	# objects to tabulate
        $nicknames,     # nickname hashref
        $base_fc,       # base frequency class
        $total,		# number of possible objects in this class
        $progress,	# indicate progress?
        $logdata->{"stats"}->{"objects"},	# total object count
        undef,		# maximum number of objects to output
        $columns,	# columns mode
        $indicate_interesting, # indicate interesting objects?
    ); 

    # output page footer
    output_footer(
        $CENSUSFILE,	# filehandle
    );
    
    # close file
    close $CENSUSFILE
        or warn "Cannot close $filename: $!\n";
    
    # one file written
    return 1;
}

# write the main census file
sub output_census_file {
    my ($logdata, $nicknames, $block_fc) = @_;
    
    # current time
    my $starttime = time;
    
    print dotted_msg("Writing census.html");
    
    output_census_page(
        "census.html",		# filename
        "All objects",		# title
        "",			# HTML snippet
        $logdata,		# log data
        $logdata->{"flat"},	# data to tabulate
        $nicknames,		# nicknames hashref
        $block_fc,		# block frequency class
        undef,			# number of possible objects in class
        $opts{'verbose'},	# indicate progress?
        1,			# don't include group FC and 1/f
        0,			# don't indicate interesting objects
    );
    
    # FIXME
    if($opts{'verbose'}) {
        say   "";
        print spaced_msg("");
    }
    
    say timestamped_msg(
        colored(
            pluralify(
                scalar(keys %{ $logdata->{"flat"} }),
                "object",
                "objects",
                " written"
            ),
            "green"
        ), 
        time - $starttime
    );
}

# write the trophy case file
sub output_trophy_file {
    my ($logdata, $nicknames, $block_fc) = @_;
    
    # current time
    my $starttime = time;
    
    print dotted_msg("Writing trophy.html");

    open my $TROPHYFILE, ">", "$opts{'file-path'}/trophy.html"
        or die "Cannot open trophy.html for writing: $!\n";
    
    # output page header
    output_header(
        $TROPHYFILE,	# filehandle
        $logdata,	# log data
    );
    
    # output navigation links
    output_navigation(
        $TROPHYFILE,	# filehandle
        $logdata,	# log data
    );
    
    # save currently selected filehandle, and select output filehandle to
    # save typing
    my $OLD_FILEHANDLE = select $TROPHYFILE;

    # xs trophies, for seeing all still lives with a given population
    say '<p class="trophycase">';
    foreach my $population (sort { $a <=> $b } keys %{ $logdata->{'census'}->{'xs'} }) {
        if(scalar keys %{ $logdata->{'census'}->{'xs'}->{$population} } == ($stilllives[$population] // -1)) {
            say qq#<img src="images/trophies/xs$population.png" title="Saw all $stilllives[$population] still lives with population $population" />#;
        }
    }
    say '</p>';

    # distinct trophies, for seeing certain numbers of distinct objects
    say '<p class="trophycase">';
    foreach my $milestone (1000, 2500, 5000, 10_000, 25_000, 50_000, 100_000, 250_000, 500_000, 1_000_000) {
        if($logdata->{'stats'}->{'distinct'} >= $milestone) {
            say qq#<img src="images/trophies/distinct$milestone.png" title="Saw # . commify($milestone) . qq# distinct objects" />#;
        }
    }
    say '</p>';
    
    # restore previously-selected filehandle
    select $OLD_FILEHANDLE;
    
    # output page footer
    output_footer(
        $TROPHYFILE,	# filehandle
    );
    
    # close file
    close $TROPHYFILE
        or warn "Cannot close index.html: $!\n";
    
    # FIXME
#    if($opts{'verbose'}) {
#        say   "";
#        print spaced_msg("");
#    }
    
    say timestamped_msg(
        colored(
            "done",
            "green"
        ), 
        time - $starttime
    );
}


# write the index census file
sub output_index_file {
    my ($logdata, $nicknames, $block_fc) = @_;
    
    # current time
    my $starttime = time;
    
    print dotted_msg("Writing index.html");

    open my $INDEXFILE, ">", "$opts{'file-path'}/index.html"
        or die "Cannot open index.html for writing: $!\n";
    
    # output page header
    output_header(
        $INDEXFILE,	# filehandle
        $logdata,	# log data
    );
    
    # output navigation links
    output_navigation(
        $INDEXFILE,	# filehandle
        $logdata,	# log data
    );
    
    # save currently selected filehandle, and select output filehandle to
    # save typing
    my $OLD_FILEHANDLE = select $INDEXFILE;

    say '<h1>Graphs</h1>';
    say qq#<div class="graph-container"><img class="graph" width="$opts{"graph-width"}" height="$opts{"graph-height"}" src="images/xs.png"></div>#;
    say qq#<div class="graph-container"><img class="graph" width="$opts{"graph-width"}" height="$opts{"graph-height"}" src="images/xs_total.png"></div>#;
    say qq#<div class="graph-container"><img class="graph" width="$opts{"graph-width"}" height="$opts{"graph-height"}" src="images/xp.png"></div>#;
    say qq#<div class="graph-container"><img class="graph" width="$opts{"graph-width"}" height="$opts{"graph-height"}" src="images/xp_total.png"></div>#;
    
    say qq#<h1>$opts{"common-objects"} most common objects</h1>#;
    
    # output census table
    output_object_table(
        $INDEXFILE,    				# filehandle
        undef,         				# don't include a title
        "",					# no HTML snippet
        $logdata->{'flat'},			# objects to tabulate
        $nicknames,     			# nickname hashref
        $block_fc,      			# base frequency class
        undef,					# number of possible objects in this class
        0,					# don't indicate progress
        $logdata->{"stats"}->{"objects"},	# total object count
        $opts{"common-objects"},		# maximum number of ranks to output
        1,					# don't include group FC and 1/f
        0,					# don't indicate interesting objects
    ); 
    
    say qq#<h1>Naturally-occurring high-period (p &ge; $opts{'high-period'}) oscillators</h1>#;

    # we'll collect high-period oscillators into this flat hash.    
    my %high_p_oscillators = ();
    
    # iterate over all oscillator periods greater than four, and merge those
    # oscillators into %high_p_oscillators.
    foreach my $period (grep { $_ >= $opts{'high-period'} } keys %{ $logdata->{'census'}->{'xp'} }) {
        @high_p_oscillators{keys %{ $logdata->{'census'}->{'xp'}->{$period} }} =
            values %{ $logdata->{'census'}->{'xp'}->{$period} };
    }
    
    # output census table
    output_object_table(
        $INDEXFILE,    				# filehandle
        undef,         				# don't include a title
        "",					# no HTML snippet
        \%high_p_oscillators,			# objects to tabulate
        $nicknames,     			# nickname hashref
        $block_fc,      			# base frequency class
        undef,					# number of possible objects in this class
        0,					# don't indicate progress
        $logdata->{"stats"}->{"objects"},	# total object count
        undef,					# output unlimited ranks
        2,					# include group FC and 1/f
    ); 
    
    say qq#<h1>Naturally-occurring non-stationary patterns</h1>#;

    # we'll collect non-stationary patterns into this flat hash.    
    my %non_stationary = ();
    
    # iterate over spaceship periods and all linear-growth p's and merge
    # those objects into %non_stationary.
    foreach my $type ("xq", "yl") {
        foreach my $period (keys %{ $logdata->{'census'}->{$type} }) {
            @non_stationary{keys %{ $logdata->{'census'}->{$type}->{$period} }} =
                values %{ $logdata->{'census'}->{$type}->{$period} };
        }
    }
    
    # output census table
    output_object_table(
        $INDEXFILE,    				# filehandle
        undef,         				# don't include a title
        "",					# no HTML snippet
        \%non_stationary,			# objects to tabulate
        $nicknames,     			# nickname hashref
        $block_fc,      			# base frequency class
        undef,					# number of possible objects in this class
        0,					# don't indicate progress
        $logdata->{"stats"}->{"objects"},	# total object count
        undef,					# output unlimited ranks
        2,					# include group FC and 1/f
    );
    
    # restore previously-selected filehandle
    select $OLD_FILEHANDLE;
    
    # output page footer
    output_footer(
        $INDEXFILE,	# filehandle
    );
    
    # close file
    close $INDEXFILE
        or warn "Cannot close index.html: $!\n";
    
    # FIXME
#    if($opts{'verbose'}) {
#        say   "";
#        print spaced_msg("");
#    }
    
    say timestamped_msg(
        colored(
            "done",
            "green"
        ), 
        time - $starttime
    );
}

# write unseen objects file
sub output_unseen_file {
    my ($logdata, $nicknames, $block_fc) = @_;
    
    # current time
    my $starttime = time;
    
    print dotted_msg("Writing unseen.html");

    open my $UNSEENFILE, ">", "$opts{'file-path'}/unseen.html"
        or die "Cannot open unseen.html for writing: $!\n";
    
    # output page header
    output_header(
        $UNSEENFILE,	# filehandle
        $logdata,	# log data
    );
    
    # output navigation links
    output_navigation(
        $UNSEENFILE,	# filehandle
        $logdata,	# log data
    );
    
    # save currently selected filehandle, and select output filehandle to
    # save typing
    my $OLD_FILEHANDLE = select $UNSEENFILE;

    say '<h1>Unseen objects</h1>';
    
    # which objects appear only in the nicknames hash, but not in the log data?
    my @nickname_keys = keys %$nicknames;
    my @logdata_keys  = keys %{ $logdata->{'flat'} };
    
    #say STDERR "n=@nickname_keys";
    #say STDERR "l=@logdata_keys";
    
    my %unseen = map { $_ => 0 } get_unique([ \@nickname_keys, \@logdata_keys ]);
    
    # output census table
    output_object_table(
        $UNSEENFILE,   				# filehandle
        undef,         				# don't include a title
        "",					# no HTML snippet
        \%unseen,				# objects to tabulate
        $nicknames,     			# nickname hashref
        $block_fc,      			# base frequency class
        undef,					# number of possible objects in this class
        0,					# don't indicate progress
        $logdata->{"stats"}->{"objects"},	# total object count
        undef,					# maximum number of ranks to output
        0,					# don't include count, FC, group FC, 1/f or group 1/f
    ); 
    
    # restore previously-selected filehandle
    select $OLD_FILEHANDLE;
    
    # output page footer
    output_footer(
        $UNSEENFILE,	# filehandle
    );
    
    # close file
    close $UNSEENFILE
        or warn "Cannot close unseen.html: $!\n";
    
    # FIXME
#    if($opts{'verbose'}) {
#        say   "";
#        print spaced_msg("");
#    }
    
    say timestamped_msg(
        colored(
            "done",
            "green"
        ), 
        time - $starttime
    );
}

# write new objects file
sub output_new_file {
    my ($logdata, $nicknames, $block_fc) = @_;
    
    # current time
    my $starttime = time;
    
    print dotted_msg("Writing new.html");

    open my $NEWFILE, ">", "$opts{'file-path'}/new.html"
        or die "Cannot open new.html for writing: $!\n";
    
    # output page header
    output_header(
        $NEWFILE,	# filehandle
        $logdata,	# log data
    );
    
    # output navigation links
    output_navigation(
        $NEWFILE,	# filehandle
        $logdata,	# log data
    );
    
    # save currently selected filehandle, and select output filehandle to
    # save typing
    my $OLD_FILEHANDLE = select $NEWFILE;

    say '<h1>New objects</h1>';
    
    my @oldseen = ();
    
    # read old seen objects file; if it doesn't exist, that's fine, we'll
    # just pretend it's empty
    my $seenobjects_file = $opts{'file-path'} . "/seenobjects.txt";
    my $SEENOBJECTS;
    open $SEENOBJECTS, "<", $seenobjects_file
        and do {
        
        # read previously-seen objects
        @oldseen   = <$SEENOBJECTS>;
        chomp(@oldseen);

        close $SEENOBJECTS
            or warn "Cannot close $seenobjects_file: $!\n";
    };
    
    # objects seen in this log run
    my @seen      = keys %{ $logdata->{'flat'} };
    
    # objects in @seen but not in @oldseen are new this run
    my %newlyseen = map { $_ => $logdata->{'flat'}->{$_} } get_unique([ \@seen, \@oldseen ]);

    # output census table
    output_object_table(
        $NEWFILE,   				# filehandle
        undef,         				# don't include a title
        "",					# no HTML snippet
        \%newlyseen,				# objects to tabulate
        $nicknames,     			# nickname hashref
        $block_fc,      			# base frequency class
        undef,					# number of possible objects in this class
        0,					# don't indicate progress
        $logdata->{"stats"}->{"objects"},	# total object count
        undef,					# maximum number of ranks to output
        1,					# don't include count, FC, group FC, 1/f or group 1/f
    ); 
    
    # output page footer
    output_footer(
        $NEWFILE,	# filehandle
    );
    
    # close file
    close $NEWFILE
        or warn "Cannot close new.html: $!\n";
    
    # restore previously-selected filehandle
    select $OLD_FILEHANDLE;
    
    # FIXME
#    if($opts{'verbose'}) {
#        say   "";
#        print spaced_msg("");
#    }
    
    say timestamped_msg(
        colored(
            "done",
            "green"
        ), 
        time - $starttime
    );
}

sub output_object_list {
    my ($objects, $type, $nicknames, $title_prefix) = @_;

    print "<li><p>$title_prefix $typenames{$type}";
            
    # compute total number of objects for this type, and output if >1
    my $numobjects = sum0 map { scalar @{ $objects->{$_} } } keys %$objects;
    if($numobjects > 1) {
        print " ($numobjects total)";
    }
            
    say ":</p>";
    say "<ul>";
            
    foreach my $p (sort {$a <=> $b } keys %$objects) {
    
        my $objects_p = $objects->{$p};
        
        # skip empty groups
        next unless scalar @$objects_p;
                
        print "<li>";
        if(exists $pnames{$type}) {
            print ucfirst($pnames{$type}) . " ";
        }
        print $p;
                
        # more than one new object of this type and p?
        if(scalar @$objects_p > 1) {
            # yes; output a sublist
                    
            say " (" . (scalar @$objects_p) . " total):";
            say "<ul>";
            say map {
                "<li>" . make_object_link($_, $nicknames, 1) . "</li>" # 1: indicate interesting objects
            } sort { $a cmp $b } @{ $objects->{$p} };
            say "</ul>";
                    
        } else {
            # no; just output this one object
                    
            say ":";
            say make_object_link($objects_p->[0], $nicknames, 1); # 1: indicate interesting objects
                    
        }
                
        say "</li>";
    }
            
    say "</ul>";
    say "</li>";
}

sub output_diary {
    my ($logdata, $nicknames) = @_;
    
    my $starttime     = time;	# current time
    my $files_written = 0;	# number of files written
    
    # number of characters written to terminal (used for line breaks).  to
    # start, the equivalent of one dotted_msg.
    my $chars_written = $opts{"output-indent"} + 1;
    
    print dotted_msg("Writing diary");
    
    # iterate through all dates, and skip days for which no data was submitted
    foreach my $date (sort { $a cmp $b } grep { exists $logdata->{'diary'}->{$_}->{'census'} } keys %{ $logdata->{'diary'} }) {
        
        open my $DIARYFILE, ">", "$opts{'file-path'}/diary${date}.html"
            or die "Cannot open diary${date}.html for writing: $!\n";
    
        # output page header
        output_header(
            $DIARYFILE,	# filehandle
            $logdata,	# log data
        );
    
        # output navigation links
        output_navigation(
            $DIARYFILE,	# filehandle
            $logdata,	# log data
        );
    
        # save currently selected filehandle, and select output filehandle to
        # save typing
        my $OLD_FILEHANDLE = select $DIARYFILE;
    
        say "<h1>Diary: $date</h1>";
        
        print "<p>Stats for the day: ";
        print totals($logdata->{'diary'}->{$date});
        say   "</p>";
        
        if($logdata->{'stats'}->{'diarylogs'} < $logdata->{'stats'}->{'logs'}) {
            say "<p>Note: diary is based on data from $logdata->{'stats'}->{'diarylogs'} logs out of $logdata->{'stats'}->{'logs'} total.</p>";
        }
        
        say "<h2>Interesting objects</h2>";
        
        say "<ul>";
        
        my $dailycensus = $logdata->{'diary'}->{$date}->{'census'};

        # go through some gymnastics to filter out the standard ships while
        # also massaging the data into the right shape, all without
        # modifying $logdata itself.
        my %dailyxqs    = map  { $_ => [
                                grep { !exists $standard_spaceships{$_} } 
                                keys %{ $dailycensus->{'xq'}->{$_} }
                            ] } 
                          keys %{ $dailycensus->{'xq'} };
        
        # count spaceship types                          
        my $xqtypecount = sum0 map { scalar @{ $dailyxqs{$_} } } keys %dailyxqs;
        
        if($xqtypecount) {
            output_object_list(\%dailyxqs, 'xq', $nicknames, "Interesting");
        }

        # same thing, for oscillators        
        my %dailyxps    = map  { $_ => [
                                grep { !exists $standard_oscillators{$_} } 
                                keys %{ $dailycensus->{'xp'}->{$_} } 
                            ] }
                          grep { $_ >= $opts{'high-period'} }
                          keys %{ $dailycensus->{'xp'} };
                          
        # count oscillator types                          
        my $xptypecount = sum0 map { scalar @{ $dailyxps{$_} } } keys %dailyxps;
        
        if($xptypecount) {
            output_object_list(\%dailyxps, 'xp', $nicknames, "Interesting");
        }
        
        # same thing yet again, for still lifes
        my %dailyxss    = map  { $_ => [
                                grep { !exists $standard_stilllifes{$_} } 
                                keys %{ $dailycensus->{'xs'}->{$_} } 
                            ] }
                          grep { $_ >= 30 } # FIXME: make configurable
                          keys %{ $dailycensus->{'xs'} };
                          
        # count oscillator types                          
        my $xstypecount = sum0 map { scalar @{ $dailyxss{$_} } } keys %dailyxss;
        
        if($xstypecount) {
            output_object_list(\%dailyxss, 'xs', $nicknames, "Interesting");
        }
        
        say '</ul>';
                            
        say "<h2>New objects</h2>";
        
        say '<ul>';

        # save a reference, save keystrokes        
        my $newobjects = $logdata->{'diary'}->{$date}->{'newobjects'};
        foreach my $type (sort { $a cmp $b } keys %$newobjects) {
            output_object_list($newobjects->{$type}, $type, $nicknames, "New");
        }
        
        say "</ul>";

        # output page footer
        output_footer(
            $DIARYFILE,	# filehandle
        );
    
        # close file
        close $DIARYFILE
            or warn "Cannot close diary.html: $!\n";
    
        # restore previously-selected filehandle
        select $OLD_FILEHANDLE;
    
        # if in verbose mode, output object type.
        if($opts{"verbose"}) {
            $chars_written = indented_print("$date ", "bright_blue", $chars_written);
        }
        
        $files_written++;
    }
    
    # if in verbose mode, and if files were written, output a new line.
    if($opts{"verbose"} && $files_written) {
        say   "";
        print spaced_msg("");
    }

    say timestamped_msg(colored(pluralify($files_written, "file", "files", " written"), "green"), time - $starttime);
}

sub output_diary_stats_page {
    my ($logdata) = @_;
    
    # current time
    my $starttime = time;
    
    print dotted_msg("Writing dailystats.html");

    open my $STATSFILE, ">", "$opts{'file-path'}/dailystats.html"
        or die "Cannot open dailystats.html for writing: $!\n";
    
    # output page header
    output_header(
        $STATSFILE,	# filehandle
        $logdata,	# log data
    );
    
    # output navigation links
    output_navigation(
        $STATSFILE,	# filehandle
        $logdata,	# log data
    );
    
    # save currently selected filehandle, and select output filehandle to
    # save typing
    my $OLD_FILEHANDLE = select $STATSFILE;

    say '<h1>Daily stats charts</h1>';
    say qq#<div class="graph-container"><img class="graph" width="$opts{"graph-width"}" height="$opts{"graph-height"}" src="images/dailylogs.png"></div>#;
    say qq#<div class="graph-container"><img class="graph" width="$opts{"graph-width"}" height="$opts{"graph-height"}" src="images/dailysoups.png"></div>#;
    say qq#<div class="graph-container"><img class="graph" width="$opts{"graph-width"}" height="$opts{"graph-height"}" src="images/dailysoupsperhaul.png"></div>#;
    say qq#<div class="graph-container"><img class="graph" width="$opts{"graph-width"}" height="$opts{"graph-height"}" src="images/dailyobjects.png"></div>#;
    say qq#<div class="graph-container"><img class="graph" width="$opts{"graph-width"}" height="$opts{"graph-height"}" src="images/dailydistinct.png"></div>#;
    say qq#<div class="graph-container"><img class="graph" width="$opts{"graph-width"}" height="$opts{"graph-height"}" src="images/dailynew.png"></div>#;
    say qq#<div class="graph-container"><img class="graph" width="$opts{"graph-width"}" height="$opts{"graph-height"}" src="images/dailynewxss.png"></div>#;
    say qq#<div class="graph-container"><img class="graph" width="$opts{"graph-width"}" height="$opts{"graph-height"}" src="images/dailynewxps.png"></div>#;
    say qq#<div class="graph-container"><img class="graph" width="$opts{"graph-width"}" height="$opts{"graph-height"}" src="images/dailynewxqs.png"></div>#;
    say qq#<div class="graph-container"><img class="graph" width="$opts{"graph-width"}" height="$opts{"graph-height"}" src="images/dailynewyls.png"></div>#;
    say qq#<div class="graph-container"><img class="graph" width="$opts{"graph-width"}" height="$opts{"graph-height"}" src="images/dailyobjectspersoup.png"></div>#;
    
    # restore previously-selected filehandle
    select $OLD_FILEHANDLE;
    
    # output page footer
    output_footer(
        $STATSFILE,	# filehandle
    );
    
    # close file
    close $STATSFILE
        or warn "Cannot close index.html: $!\n";
    
    # FIXME
    # EDIT: fix what?
#    if($opts{'verbose'}) {
#        say   "";
#        print spaced_msg("");
#    }
    
    say timestamped_msg(
        colored(
            "done",
            "green"
        ), 
        time - $starttime
    );
}

# output object type files (xq4, xp2, xs40, yl384, ...)
sub output_type_files {
    my ($logdata, $nicknames, $block_fc) = @_;

    my $starttime     = time;	# current time
    my $files_written = 0;	# number of files written
    
    # number of characters written to terminal (used for line breaks).  to
    # start, the equivalent of one dotted_msg.
    my $chars_written = $opts{"output-indent"} + 1;
    
    print dotted_msg("Writing census files");

    # iterate over all encountered object types.
    foreach my $type (sort keys %{ $logdata->{"census"} }) {
        foreach my $p (sort { $a <=> $b } keys %{ $logdata->{"census"}->{$type} }) {

            # Catagolue link for this type.
            my $link = "$opts{'catagolue-url'}/census/b3s23/C1/$type$p";
            
            my $total = ($type eq "xs") ? $stilllives[$p] : undef;

            # write file and count it.
            $files_written += output_census_page(
                "$type$p.html",				# filename
                qq#<a href="$link">$type$p</a>#,	# title
                qq#<center><p><img src="images/dailynew${type}${p}s.png" /></p></center>#,	# extra HTML snippet
                $logdata,				# log data
                $logdata->{"census"}->{$type}->{$p},	# data to tabulate
                $nicknames,				# nicknames hashref
                $block_fc,				# block frequency class
                $total,					# total number of elements in this class
                0,					# don't indicate progress
                2,					# include group FC and 1/f
                0,					# don't indicate interesting objects
            );
            
            # if in verbose mode, output object type.
            if($opts{"verbose"}) {
                $chars_written = indented_print("$type$p ", "bright_blue", $chars_written);
            }            
        }
    }

    # if in verbose mode, and if files were written, output a new line.
    if($opts{"verbose"} && $files_written) {
        say   "";
        print spaced_msg("");
    }

    say timestamped_msg(colored(pluralify($files_written, "file", "files", " written"), "green"), time - $starttime);
}

# dump seen objects into a text file so we can figure out what's new next
# time this script is run
sub dump_seen_objects {
    my ($logdata) = @_;
    
    my $filename = $opts{'file-path'} . "/seenobjects.txt";
    
    open my $SEENOBJECTS, ">", $filename
        or die "Cannot open $filename: $!\n";
        
    say $SEENOBJECTS join "\n", keys %{ $logdata->{'flat'} };
    
    close $SEENOBJECTS
        or warn "Cannot close $filename: $!\n";

}
